// Configuration
const API_BASE_URL = 'https://sensorsnodered.kalpatech.co.in';
const RELAY_COUNT = 5;
const RELAY_NAMES = ['Relay 1', 'Relay 2', 'Relay 3', 'Relay 4', 'Relay 5'];

// Global state
let relayData = [];
let currentEditingRelay = null;
let updateInterval = null;

// Initialize on page load
document.addEventListener('DOMContentLoaded', () => {
    initializeDashboard();
    startAutoUpdate();
});

// Initialize dashboard
async function initializeDashboard() {
    try {
        await fetchRelayStatus();
        updateSystemTime();
        setInterval(updateSystemTime, 1000);
    } catch (error) {
        console.error('Initialization error:', error);
        showConnectionError();
    }
}

// Fetch relay status from Node-RED
async function fetchRelayStatus() {
    try {
        const response = await fetch(`${API_BASE_URL}/api/relays/status`);
        if (!response.ok) throw new Error('Failed to fetch relay status');
        
        const data = await response.json();
        relayData = data.relays || [];
        
        updateDashboard(data);
        updateConnectionStatus(true);
    } catch (error) {
        console.error('Error fetching relay status:', error);
        updateConnectionStatus(false);
        throw error;
    }
}

// Update dashboard UI
function updateDashboard(data) {
    // Update RTC display
    if (data.rtc) {
        document.getElementById('rtcDate').textContent = data.rtc.date || '--/--/----';
        document.getElementById('rtcTime').textContent = data.rtc.time || '--:--:--';
    }
    
    // Update stats
    updateStats(data.relays);
    
    // Render relay cards
    renderRelayCards(data.relays);
}

// Update statistics
function updateStats(relays) {
    if (!relays) return;
    
    const activeCount = relays.filter(r => r.state).length;
    const autoCount = relays.filter(r => r.mode === 'Auto').length;
    const manualCount = relays.filter(r => r.mode !== 'Auto').length;
    
    document.getElementById('activeRelays').textContent = `${activeCount}/${RELAY_COUNT}`;
    document.getElementById('autoModeCount').textContent = autoCount;
    document.getElementById('manualModeCount').textContent = manualCount;
    
    const now = new Date();
    document.getElementById('lastUpdate').textContent = 
        `${now.getHours().toString().padStart(2, '0')}:${now.getMinutes().toString().padStart(2, '0')}`;
}

// Render relay cards
function renderRelayCards(relays) {
    const container = document.getElementById('relayContainer');
    container.innerHTML = '';
    
    if (!relays || relays.length === 0) {
        container.innerHTML = '<div class="col-12"><div class="alert alert-warning">No relay data available</div></div>';
        return;
    }
    
    relays.forEach((relay, index) => {
        const card = createRelayCard(relay, index);
        container.appendChild(card);
    });
}

// Create individual relay card
function createRelayCard(relay, index) {
    const col = document.createElement('div');
    col.className = 'col-lg-6 col-xl-4';
    
    const stateClass = relay.state ? 'relay-state-on' : 'relay-state-off';
    const stateText = relay.state ? 'ON' : 'OFF';
    const stateIcon = relay.state ? 'power' : 'power';
    
    col.innerHTML = `
        <div class="relay-card">
            <div class="relay-header">
                <h5 class="relay-title">
                    <i class="bi bi-lightning-charge-fill"></i>
                    ${RELAY_NAMES[index]}
                </h5>
                <span class="relay-state-badge ${stateClass}">
                    <i class="bi bi-${stateIcon}"></i> ${stateText}
                </span>
            </div>
            
            <div class="relay-controls">
                <button class="mode-btn btn btn-outline-primary ${relay.mode === 'Auto' ? 'active' : ''}" 
                        onclick="setRelayMode(${index}, 'auto')">
                    <i class="bi bi-robot"></i> Auto
                </button>
                <button class="mode-btn btn btn-outline-success ${relay.mode === 'Manual On' ? 'active' : ''}" 
                        onclick="setRelayMode(${index}, 'manual_on')">
                    <i class="bi bi-toggle-on"></i> Manual ON
                </button>
                <button class="mode-btn btn btn-outline-secondary ${relay.mode === 'Manual Off' ? 'active' : ''}" 
                        onclick="setRelayMode(${index}, 'manual_off')">
                    <i class="bi bi-toggle-off"></i> Manual OFF
                </button>
            </div>
            
            <div class="routines-section">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h6 class="mb-0"><i class="bi bi-clock-history"></i> Scheduled Routines</h6>
                    <button class="btn btn-sm btn-primary" onclick="openRoutineModal(${index})">
                        <i class="bi bi-pencil"></i> Edit
                    </button>
                </div>
                <div id="routines-${index}">
                    ${renderRoutines(relay.routines)}
                </div>
            </div>
        </div>
    `;
    
    return col;
}

// Render routines list
function renderRoutines(routines) {
    if (!routines || routines.length === 0) {
        return '<div class="text-muted text-center py-3">No routines configured</div>';
    }
    
    return routines.map((routine, idx) => {
        const disabledClass = routine.enabled ? '' : 'disabled';
        const statusIcon = routine.enabled ? 'check-circle-fill text-success' : 'x-circle-fill text-muted';
        
        return `
            <div class="routine-item ${disabledClass}">
                <div class="routine-time">
                    <i class="bi bi-${statusIcon}"></i>
                    <span><i class="bi bi-sun"></i> ON: ${routine.on}</span>
                    <span><i class="bi bi-moon"></i> OFF: ${routine.off}</span>
                </div>
            </div>
        `;
    }).join('');
}

// Set relay mode
async function setRelayMode(relayIndex, mode) {
    try {
        const response = await fetch(`${API_BASE_URL}/api/relays/mode`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                relay: relayIndex,
                mode: mode
            })
        });
        
        if (!response.ok) throw new Error('Failed to set relay mode');
        
        // Refresh status after a short delay
        setTimeout(fetchRelayStatus, 500);
        
        showNotification('success', `Relay ${relayIndex + 1} mode set to ${mode}`);
    } catch (error) {
        console.error('Error setting relay mode:', error);
        showNotification('error', 'Failed to set relay mode');
    }
}

// Open routine editing modal
function openRoutineModal(relayIndex) {
    currentEditingRelay = relayIndex;
    const relay = relayData[relayIndex];
    
    document.getElementById('modalRelayName').textContent = RELAY_NAMES[relayIndex];
    
    const routinesList = document.getElementById('routinesList');
    routinesList.innerHTML = '';
    
    if (relay && relay.routines) {
        relay.routines.forEach((routine, idx) => {
            routinesList.appendChild(createRoutineInput(routine, idx));
        });
    }
    
    const modal = new bootstrap.Modal(document.getElementById('routineModal'));
    modal.show();
}

// Create routine input form
function createRoutineInput(routine, index) {
    const div = document.createElement('div');
    div.className = 'routine-input-group';
    div.id = `routine-${index}`;
    
    const onTime = routine.on || '06:00';
    const offTime = routine.off || '18:00';
    const enabled = routine.enabled !== false;
    
    div.innerHTML = `
        <div class="d-flex justify-content-between align-items-center mb-3">
            <h6 class="mb-0">Routine ${index + 1}</h6>
            <div class="d-flex gap-2 align-items-center">
                <div class="form-check form-switch">
                    <input class="form-check-input" type="checkbox" id="enabled-${index}" 
                           ${enabled ? 'checked' : ''} onchange="toggleRoutineEnabled(${index})">
                    <label class="form-check-label" for="enabled-${index}">Enabled</label>
                </div>
                <button class="btn btn-sm btn-danger" onclick="deleteRoutine(${index})">
                    <i class="bi bi-trash"></i>
                </button>
            </div>
        </div>
        <div class="row g-3">
            <div class="col-md-6">
                <label class="form-label"><i class="bi bi-sun"></i> ON Time</label>
                <input type="time" class="form-control time-input" id="on-${index}" value="${onTime}">
            </div>
            <div class="col-md-6">
                <label class="form-label"><i class="bi bi-moon"></i> OFF Time</label>
                <input type="time" class="form-control time-input" id="off-${index}" value="${offTime}">
            </div>
        </div>
    `;
    
    return div;
}

// Toggle routine enabled state
function toggleRoutineEnabled(index) {
    const routineDiv = document.getElementById(`routine-${index}`);
    const checkbox = document.getElementById(`enabled-${index}`);
    
    if (checkbox.checked) {
        routineDiv.classList.remove('disabled');
    } else {
        routineDiv.classList.add('disabled');
    }
}

// Add new routine
function addRoutine() {
    const routinesList = document.getElementById('routinesList');
    const currentCount = routinesList.children.length;
    
    if (currentCount >= 10) {
        showNotification('warning', 'Maximum 10 routines allowed per relay');
        return;
    }
    
    const newRoutine = {
        on: '06:00',
        off: '18:00',
        enabled: true
    };
    
    routinesList.appendChild(createRoutineInput(newRoutine, currentCount));
}

// Delete routine
function deleteRoutine(index) {
    const routineDiv = document.getElementById(`routine-${index}`);
    if (routineDiv && confirm('Are you sure you want to delete this routine?')) {
        routineDiv.remove();
        
        // Re-index remaining routines
        const routinesList = document.getElementById('routinesList');
        Array.from(routinesList.children).forEach((child, newIndex) => {
            child.id = `routine-${newIndex}`;
            // Update all IDs within the routine
            const inputs = child.querySelectorAll('[id]');
            inputs.forEach(input => {
                const oldId = input.id;
                const prefix = oldId.split('-')[0];
                input.id = `${prefix}-${newIndex}`;
            });
        });
    }
}

// Save routines
async function saveRoutines() {
    const routinesList = document.getElementById('routinesList');
    const routines = [];
    
    Array.from(routinesList.children).forEach((routineDiv, index) => {
        const onTime = document.getElementById(`on-${index}`).value;
        const offTime = document.getElementById(`off-${index}`).value;
        const enabled = document.getElementById(`enabled-${index}`).checked;
        
        routines.push({
            on: onTime,
            off: offTime,
            enabled: enabled
        });
    });
    
    try {
        const response = await fetch(`${API_BASE_URL}/api/relays/routines`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                relay: currentEditingRelay,
                routines: routines
            })
        });
        
        if (!response.ok) throw new Error('Failed to save routines');
        
        // Close modal
        const modal = bootstrap.Modal.getInstance(document.getElementById('routineModal'));
        modal.hide();
        
        // Refresh status
        setTimeout(fetchRelayStatus, 500);
        
        showNotification('success', 'Routines saved successfully');
    } catch (error) {
        console.error('Error saving routines:', error);
        showNotification('error', 'Failed to save routines');
    }
}

// Sync RTC with system time
async function syncRTCWithSystem() {
    const now = new Date();
    const date = now.toISOString().split('T')[0]; // YYYY-MM-DD
    const time = `${now.getHours().toString().padStart(2, '0')}:${now.getMinutes().toString().padStart(2, '0')}`;
    
    try {
        const response = await fetch(`${API_BASE_URL}/api/rtc/sync`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ date, time })
        });
        
        if (!response.ok) throw new Error('Failed to sync RTC');
        
        showNotification('success', 'RTC synchronized with system time');
        setTimeout(fetchRelayStatus, 1000);
    } catch (error) {
        console.error('Error syncing RTC:', error);
        showNotification('error', 'Failed to sync RTC');
    }
}

// Update system time display
function updateSystemTime() {
    const now = new Date();
    document.getElementById('currentTime').textContent = 
        now.toLocaleTimeString('en-US', { hour12: false });
}

// Update connection status
function updateConnectionStatus(connected) {
    const statusBadge = document.getElementById('connectionStatus');
    if (connected) {
        statusBadge.className = 'badge bg-success me-2';
        statusBadge.innerHTML = '<i class="bi bi-wifi"></i> Connected';
    } else {
        statusBadge.className = 'badge bg-danger me-2';
        statusBadge.innerHTML = '<i class="bi bi-wifi-off"></i> Disconnected';
    }
}

// Show connection error
function showConnectionError() {
    const container = document.getElementById('relayContainer');
    container.innerHTML = `
        <div class="col-12">
            <div class="alert alert-danger">
                <h5><i class="bi bi-exclamation-triangle"></i> Connection Error</h5>
                <p>Unable to connect to the relay controller. Please check your connection and try again.</p>
                <button class="btn btn-danger" onclick="fetchRelayStatus()">
                    <i class="bi bi-arrow-repeat"></i> Retry
                </button>
            </div>
        </div>
    `;
}

// Show notification
function showNotification(type, message) {
    // Create a toast notification
    const toastContainer = document.getElementById('toastContainer') || createToastContainer();
    
    const toastId = `toast-${Date.now()}`;
    const bgClass = type === 'success' ? 'bg-success' : type === 'error' ? 'bg-danger' : 'bg-warning';
    const icon = type === 'success' ? 'check-circle' : type === 'error' ? 'x-circle' : 'exclamation-circle';
    
    const toastHTML = `
        <div id="${toastId}" class="toast align-items-center text-white ${bgClass} border-0" role="alert">
            <div class="d-flex">
                <div class="toast-body">
                    <i class="bi bi-${icon}"></i> ${message}
                </div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        </div>
    `;
    
    toastContainer.insertAdjacentHTML('beforeend', toastHTML);
    
    const toastElement = document.getElementById(toastId);
    const toast = new bootstrap.Toast(toastElement, { autohide: true, delay: 3000 });
    toast.show();
    
    toastElement.addEventListener('hidden.bs.toast', () => {
        toastElement.remove();
    });
}

// Create toast container
function createToastContainer() {
    const container = document.createElement('div');
    container.id = 'toastContainer';
    container.className = 'toast-container position-fixed bottom-0 end-0 p-3';
    container.style.zIndex = '1100';
    document.body.appendChild(container);
    return container;
}

// Start auto-update
function startAutoUpdate() {
    // Update every 5 seconds
    updateInterval = setInterval(fetchRelayStatus, 5000);
}

// Stop auto-update
function stopAutoUpdate() {
    if (updateInterval) {
        clearInterval(updateInterval);
        updateInterval = null;
    }
}

// Handle page visibility changes
document.addEventListener('visibilitychange', () => {
    if (document.hidden) {
        stopAutoUpdate();
    } else {
        startAutoUpdate();
        fetchRelayStatus();
    }
});
