// ============================================
// CONFIGURATION & CONSTANTS
// ============================================
const CONFIG = {
    API_BASE_URL: 'https://sensorsnodered.kalpatech.co.in',
    RELAY_COUNT: 5,
    RELAY_NAMES: ['Relay 1', 'Relay 2', 'Relay 3', 'Relay 4', 'Relay 5'],
    UPDATE_INTERVAL: 5000, // 5 seconds
    RETRY_DELAY: 3000, // 3 seconds
    MAX_RETRIES: 3
};

// ============================================
// GLOBAL STATE
// ============================================
let state = {
    relayData: [],
    currentEditingRelay: null,
    updateInterval: null,
    isConnected: false,
    retryCount: 0,
    darkMode: false
};

// ============================================
// INITIALIZATION
// ============================================
document.addEventListener('DOMContentLoaded', () => {
    initializeApp();
});

async function initializeApp() {
    try {
        // Load saved preferences
        loadPreferences();
        
        // Initialize system time
        updateSystemTime();
        setInterval(updateSystemTime, 1000);
        
        // Fetch initial data
        await fetchRelayStatus();
        
        // Hide loading overlay
        hideLoadingOverlay();
        
        // Start auto-update
        startAutoUpdate();
        
        // Setup event listeners
        setupEventListeners();
        
        showToast('success', 'Dashboard loaded successfully!');
    } catch (error) {
        console.error('Initialization error:', error);
        hideLoadingOverlay();
        showConnectionError();
    }
}

// ============================================
// PREFERENCES MANAGEMENT
// ============================================
function loadPreferences() {
    const savedDarkMode = localStorage.getItem('darkMode');
    if (savedDarkMode === 'true') {
        enableDarkMode();
    }
}

function toggleDarkMode() {
    state.darkMode = !state.darkMode;
    localStorage.setItem('darkMode', state.darkMode);
    
    if (state.darkMode) {
        enableDarkMode();
    } else {
        disableDarkMode();
    }
}

function enableDarkMode() {
    document.body.classList.add('dark-mode');
    state.darkMode = true;
}

function disableDarkMode() {
    document.body.classList.remove('dark-mode');
    state.darkMode = false;
}

// ============================================
// UI UTILITIES
// ============================================
function hideLoadingOverlay() {
    const overlay = document.getElementById('loadingOverlay');
    if (overlay) {
        overlay.classList.add('hidden');
        setTimeout(() => overlay.style.display = 'none', 300);
    }
}

function showLoadingOverlay() {
    const overlay = document.getElementById('loadingOverlay');
    if (overlay) {
        overlay.style.display = 'flex';
        setTimeout(() => overlay.classList.remove('hidden'), 10);
    }
}

// ============================================
// DATA FETCHING
// ============================================
async function fetchRelayStatus() {
    try {
        const response = await fetch(`${CONFIG.API_BASE_URL}/api/relays/status`, {
            headers: {
                'Accept': 'application/json',
                'Cache-Control': 'no-cache'
            }
        });
        
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        const data = await response.json();
        state.relayData = data.relays || [];
        
        updateDashboard(data);
        updateConnectionStatus(true);
        state.retryCount = 0;
        
    } catch (error) {
        console.error('Error fetching relay status:', error);
        updateConnectionStatus(false);
        
        if (state.retryCount < CONFIG.MAX_RETRIES) {
            state.retryCount++;
            setTimeout(fetchRelayStatus, CONFIG.RETRY_DELAY);
        } else {
            showConnectionError();
        }
        
        throw error;
    }
}

async function refreshStatus() {
    const btn = event.target.closest('button');
    const icon = btn.querySelector('i');
    
    // Add rotation animation
    icon.classList.add('rotate');
    btn.disabled = true;
    
    try {
        await fetchRelayStatus();
        showToast('success', 'Status refreshed successfully');
    } catch (error) {
        showToast('error', 'Failed to refresh status');
    } finally {
        setTimeout(() => {
            icon.classList.remove('rotate');
            btn.disabled = false;
        }, 500);
    }
}

// ============================================
// DASHBOARD UPDATE
// ============================================
function updateDashboard(data) {
    // Update RTC display
    updateRTCDisplay(data.rtc);
    
    // Update statistics
    updateStatistics(data.relays);
    
    // Render relay cards
    renderRelayCards(data.relays);
}

function updateRTCDisplay(rtc) {
    if (!rtc) return;
    
    const dateElement = document.getElementById('rtcDate');
    const timeElement = document.getElementById('rtcTime');
    
    if (dateElement && rtc.date) {
        dateElement.textContent = rtc.date;
        dateElement.classList.add('fade-in');
    }
    
    if (timeElement && rtc.time) {
        timeElement.textContent = rtc.time;
        timeElement.classList.add('fade-in');
    }
}

function updateStatistics(relays) {
    if (!relays || relays.length === 0) return;
    
    const activeCount = relays.filter(r => r.state).length;
    const autoCount = relays.filter(r => r.mode === 'Auto').length;
    const manualCount = relays.filter(r => r.mode !== 'Auto').length;
    
    animateValue('activeRelays', `${activeCount}/${CONFIG.RELAY_COUNT}`);
    animateValue('autoModeCount', autoCount);
    animateValue('manualModeCount', manualCount);
    
    const now = new Date();
    const timeStr = `${now.getHours().toString().padStart(2, '0')}:${now.getMinutes().toString().padStart(2, '0')}`;
    document.getElementById('lastUpdate').textContent = timeStr;
}

function animateValue(elementId, value) {
    const element = document.getElementById(elementId);
    if (!element) return;
    
    element.style.opacity = '0';
    element.style.transform = 'scale(0.8)';
    
    setTimeout(() => {
        element.textContent = value;
        element.style.transition = 'all 0.3s ease';
        element.style.opacity = '1';
        element.style.transform = 'scale(1)';
    }, 150);
}

// ============================================
// RELAY CARDS RENDERING
// ============================================
function renderRelayCards(relays) {
    const container = document.getElementById('relayContainer');
    if (!container) return;
    
    if (!relays || relays.length === 0) {
        container.innerHTML = createEmptyState();
        return;
    }
    
    container.innerHTML = '';
    
    relays.forEach((relay, index) => {
        const card = createRelayCard(relay, index);
        container.appendChild(card);
        
        // Add staggered animation
        setTimeout(() => {
            card.classList.add('fade-in');
        }, index * 100);
    });
}

function createRelayCard(relay, index) {
    const col = document.createElement('div');
    col.className = 'col-12 col-md-6 col-lg-4';
    col.style.opacity = '0';
    
    const isOn = relay.state;
    const stateClass = isOn ? 'relay-state-on' : 'relay-state-off';
    const stateText = isOn ? 'ON' : 'OFF';
    const stateIcon = isOn ? 'power' : 'power';
    const cardActiveClass = isOn ? 'active' : '';
    
    col.innerHTML = `
        <div class="relay-card ${cardActiveClass}">
            <div class="relay-header">
                <h5 class="relay-title">
                    <i class="bi bi-lightning-charge-fill"></i>
                    ${CONFIG.RELAY_NAMES[index]}
                </h5>
                <span class="relay-state-badge ${stateClass}">
                    <i class="bi bi-${stateIcon}"></i>
                    <span>${stateText}</span>
                </span>
            </div>
            
            <div class="relay-controls">
                <button class="mode-btn btn btn-outline-primary ${relay.mode === 'Auto' ? 'active' : ''}" 
                        onclick="setRelayMode(${index}, 'auto')"
                        aria-label="Set ${CONFIG.RELAY_NAMES[index]} to Auto mode">
                    <i class="bi bi-robot"></i>
                    <span>Auto</span>
                </button>
                <button class="mode-btn btn btn-outline-success ${relay.mode === 'Manual On' ? 'active' : ''}" 
                        onclick="setRelayMode(${index}, 'manual_on')"
                        aria-label="Set ${CONFIG.RELAY_NAMES[index]} to Manual ON">
                    <i class="bi bi-toggle-on"></i>
                    <span>ON</span>
                </button>
                <button class="mode-btn btn btn-outline-secondary ${relay.mode === 'Manual Off' ? 'active' : ''}" 
                        onclick="setRelayMode(${index}, 'manual_off')"
                        aria-label="Set ${CONFIG.RELAY_NAMES[index]} to Manual OFF">
                    <i class="bi bi-toggle-off"></i>
                    <span>OFF</span>
                </button>
            </div>
            
            <div class="routines-section">
                <div class="routines-header">
                    <h6>
                        <i class="bi bi-clock-history me-2"></i>
                        Scheduled Routines
                    </h6>
                    <button class="btn btn-sm btn-primary" onclick="openRoutineModal(${index})"
                            aria-label="Edit routines for ${CONFIG.RELAY_NAMES[index]}">
                        <i class="bi bi-pencil-fill"></i>
                        <span class="d-none d-sm-inline ms-1">Edit</span>
                    </button>
                </div>
                <div id="routines-${index}">
                    ${renderRoutines(relay.routines)}
                </div>
            </div>
        </div>
    `;
    
    return col;
}

function renderRoutines(routines) {
    if (!routines || routines.length === 0) {
        return `
            <div class="text-center text-muted py-3">
                <i class="bi bi-calendar-x fs-3 d-block mb-2"></i>
                <small>No routines configured</small>
            </div>
        `;
    }
    
    return routines.map((routine, idx) => {
        const disabledClass = routine.enabled ? '' : 'disabled';
        const statusIcon = routine.enabled ? 'check-circle-fill text-success' : 'x-circle-fill text-muted';
        
        return `
            <div class="routine-item ${disabledClass}">
                <div class="routine-time">
                    <i class="bi bi-${statusIcon}"></i>
                    <span>
                        <i class="bi bi-sunrise"></i>
                        <strong>ON:</strong> ${routine.on}
                    </span>
                    <span>
                        <i class="bi bi-sunset"></i>
                        <strong>OFF:</strong> ${routine.off}
                    </span>
                </div>
            </div>
        `;
    }).join('');
}

function createEmptyState() {
    return `
        <div class="col-12">
            <div class="text-center py-5">
                <i class="bi bi-inbox fs-1 text-muted mb-3 d-block"></i>
                <h5 class="text-muted">No relay data available</h5>
                <p class="text-muted">Please check your connection and try again</p>
                <button class="btn btn-primary mt-3" onclick="fetchRelayStatus()">
                    <i class="bi bi-arrow-repeat"></i> Retry
                </button>
            </div>
        </div>
    `;
}

// ============================================
// RELAY CONTROL
// ============================================
async function setRelayMode(relayIndex, mode) {
    const btn = event.target.closest('button');
    const originalHTML = btn.innerHTML;
    
    // Show loading state
    btn.disabled = true;
    btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span>';
    
    try {
        const response = await fetch(`${CONFIG.API_BASE_URL}/api/relays/mode`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            },
            body: JSON.stringify({
                relay: relayIndex,
                mode: mode
            })
        });
        
        if (!response.ok) {
            throw new Error('Failed to set relay mode');
        }
        
        // Wait a moment for ESP8266 to process
        await new Promise(resolve => setTimeout(resolve, 800));
        
        // Refresh status
        await fetchRelayStatus();
        
        const modeName = mode === 'auto' ? 'Auto' : mode === 'manual_on' ? 'Manual ON' : 'Manual OFF';
        showToast('success', `${CONFIG.RELAY_NAMES[relayIndex]} set to ${modeName}`);
        
    } catch (error) {
        console.error('Error setting relay mode:', error);
        showToast('error', `Failed to set ${CONFIG.RELAY_NAMES[relayIndex]} mode`);
    } finally {
        btn.innerHTML = originalHTML;
        btn.disabled = false;
    }
}

// ============================================
// ROUTINE MANAGEMENT
// ============================================
function openRoutineModal(relayIndex) {
    state.currentEditingRelay = relayIndex;
    const relay = state.relayData[relayIndex];
    
    document.getElementById('modalRelayName').textContent = `Edit Routines - ${CONFIG.RELAY_NAMES[relayIndex]}`;
    
    const routinesList = document.getElementById('routinesList');
    routinesList.innerHTML = '';
    
    if (relay && relay.routines && relay.routines.length > 0) {
        relay.routines.forEach((routine, idx) => {
            routinesList.appendChild(createRoutineInput(routine, idx));
        });
    } else {
        // Add one default routine
        routinesList.appendChild(createRoutineInput({ on: '06:00', off: '18:00', enabled: true }, 0));
    }
    
    const modal = new bootstrap.Modal(document.getElementById('routineModal'));
    modal.show();
}

function createRoutineInput(routine, index) {
    const div = document.createElement('div');
    div.className = 'routine-input-group';
    div.id = `routine-${index}`;
    
    const onTime = routine.on || '06:00';
    const offTime = routine.off || '18:00';
    const enabled = routine.enabled !== false;
    
    if (!enabled) {
        div.classList.add('disabled');
    }
    
    div.innerHTML = `
        <div class="routine-header">
            <h6><i class="bi bi-calendar-event"></i> Routine ${index + 1}</h6>
            <div class="d-flex gap-2 align-items-center flex-wrap">
                <div class="form-check form-switch">
                    <input class="form-check-input" type="checkbox" id="enabled-${index}" 
                           ${enabled ? 'checked' : ''} onchange="toggleRoutineEnabled(${index})">
                    <label class="form-check-label" for="enabled-${index}">
                        <small class="fw-semibold">Enabled</small>
                    </label>
                </div>
                <button class="btn btn-sm btn-danger" onclick="deleteRoutine(${index})"
                        aria-label="Delete routine ${index + 1}">
                    <i class="bi bi-trash-fill"></i>
                </button>
            </div>
        </div>
        <div class="row g-3">
            <div class="col-md-6">
                <label class="form-label fw-semibold">
                    <i class="bi bi-sunrise-fill text-warning"></i> ON Time
                </label>
                <input type="time" class="form-control time-input" id="on-${index}" value="${onTime}">
            </div>
            <div class="col-md-6">
                <label class="form-label fw-semibold">
                    <i class="bi bi-sunset-fill text-primary"></i> OFF Time
                </label>
                <input type="time" class="form-control time-input" id="off-${index}" value="${offTime}">
            </div>
        </div>
    `;
    
    return div;
}

function toggleRoutineEnabled(index) {
    const routineDiv = document.getElementById(`routine-${index}`);
    const checkbox = document.getElementById(`enabled-${index}`);
    
    if (checkbox.checked) {
        routineDiv.classList.remove('disabled');
    } else {
        routineDiv.classList.add('disabled');
    }
}

function addRoutine() {
    const routinesList = document.getElementById('routinesList');
    const currentCount = routinesList.children.length;
    
    if (currentCount >= 10) {
        showToast('warning', 'Maximum 10 routines allowed per relay');
        return;
    }
    
    const newRoutine = {
        on: '06:00',
        off: '18:00',
        enabled: true
    };
    
    const routineElement = createRoutineInput(newRoutine, currentCount);
    routinesList.appendChild(routineElement);
    
    // Scroll to new routine
    setTimeout(() => {
        routineElement.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    }, 100);
}

function deleteRoutine(index) {
    const routineDiv = document.getElementById(`routine-${index}`);
    const routinesList = document.getElementById('routinesList');
    
    if (routinesList.children.length === 1) {
        showToast('warning', 'At least one routine is required');
        return;
    }
    
    if (!confirm('Are you sure you want to delete this routine?')) {
        return;
    }
    
    // Fade out animation
    routineDiv.style.opacity = '0';
    routineDiv.style.transform = 'translateX(-20px)';
    
    setTimeout(() => {
        routineDiv.remove();
        
        // Re-index remaining routines
        Array.from(routinesList.children).forEach((child, newIndex) => {
            child.id = `routine-${newIndex}`;
            
            // Update header
            const header = child.querySelector('h6');
            if (header) {
                header.innerHTML = `<i class="bi bi-calendar-event"></i> Routine ${newIndex + 1}`;
            }
            
            // Update all IDs within the routine
            const inputs = child.querySelectorAll('[id]');
            inputs.forEach(input => {
                const oldId = input.id;
                const prefix = oldId.split('-')[0];
                input.id = `${prefix}-${newIndex}`;
                
                // Update labels
                const label = child.querySelector(`label[for="${oldId}"]`);
                if (label) {
                    label.setAttribute('for', `${prefix}-${newIndex}`);
                }
            });
            
            // Update onclick handlers
            const checkbox = child.querySelector('.form-check-input');
            if (checkbox) {
                checkbox.setAttribute('onchange', `toggleRoutineEnabled(${newIndex})`);
            }
            
            const deleteBtn = child.querySelector('.btn-danger');
            if (deleteBtn) {
                deleteBtn.setAttribute('onclick', `deleteRoutine(${newIndex})`);
            }
        });
    }, 300);
}

async function saveRoutines() {
    const routinesList = document.getElementById('routinesList');
    const routines = [];
    
    Array.from(routinesList.children).forEach((routineDiv, index) => {
        const onTime = document.getElementById(`on-${index}`).value;
        const offTime = document.getElementById(`off-${index}`).value;
        const enabled = document.getElementById(`enabled-${index}`).checked;
        
        routines.push({
            on: onTime,
            off: offTime,
            enabled: enabled
        });
    });
    
    // Show loading state
    const saveBtn = event.target;
    const originalHTML = saveBtn.innerHTML;
    saveBtn.disabled = true;
    saveBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Saving...';
    
    try {
        const response = await fetch(`${CONFIG.API_BASE_URL}/api/relays/routines`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            },
            body: JSON.stringify({
                relay: state.currentEditingRelay,
                routines: routines
            })
        });
        
        if (!response.ok) {
            throw new Error('Failed to save routines');
        }
        
        // Close modal
        const modal = bootstrap.Modal.getInstance(document.getElementById('routineModal'));
        modal.hide();
        
        // Wait a moment for ESP8266 to process
        await new Promise(resolve => setTimeout(resolve, 800));
        
        // Refresh status
        await fetchRelayStatus();
        
        showToast('success', `Routines saved for ${CONFIG.RELAY_NAMES[state.currentEditingRelay]}`);
        
    } catch (error) {
        console.error('Error saving routines:', error);
        showToast('error', 'Failed to save routines');
    } finally {
        saveBtn.innerHTML = originalHTML;
        saveBtn.disabled = false;
    }
}

// ============================================
// RTC MANAGEMENT
// ============================================
async function syncRTCWithSystem() {
    const btn = event.target.closest('button');
    const originalHTML = btn.innerHTML;
    
    btn.disabled = true;
    btn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Syncing...';
    
    try {
        const now = new Date();
        const date = now.toISOString().split('T')[0]; // YYYY-MM-DD
        const time = `${now.getHours().toString().padStart(2, '0')}:${now.getMinutes().toString().padStart(2, '0')}`;
        
        const response = await fetch(`${CONFIG.API_BASE_URL}/api/rtc/sync`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            },
            body: JSON.stringify({ date, time })
        });
        
        if (!response.ok) {
            throw new Error('Failed to sync RTC');
        }
        
        showToast('success', 'RTC synchronized with system time');
        
        // Wait and refresh
        await new Promise(resolve => setTimeout(resolve, 1000));
        await fetchRelayStatus();
        
    } catch (error) {
        console.error('Error syncing RTC:', error);
        showToast('error', 'Failed to sync RTC');
    } finally {
        btn.innerHTML = originalHTML;
        btn.disabled = false;
    }
}

// ============================================
// CONNECTION STATUS
// ============================================
function updateConnectionStatus(connected) {
    const statusBadge = document.getElementById('connectionStatus');
    if (!statusBadge) return;
    
    state.isConnected = connected;
    
    if (connected) {
        statusBadge.className = 'badge bg-success d-flex align-items-center gap-1 px-3 py-2';
        statusBadge.innerHTML = '<i class="bi bi-wifi"></i><span class="d-none d-md-inline">Connected</span>';
    } else {
        statusBadge.className = 'badge bg-danger d-flex align-items-center gap-1 px-3 py-2';
        statusBadge.innerHTML = '<i class="bi bi-wifi-off"></i><span class="d-none d-md-inline">Disconnected</span>';
    }
}

function showConnectionError() {
    const container = document.getElementById('relayContainer');
    if (!container) return;
    
    container.innerHTML = `
        <div class="col-12">
            <div class="alert alert-danger shadow-sm">
                <div class="d-flex align-items-start">
                    <i class="bi bi-exclamation-triangle-fill fs-2 me-3"></i>
                    <div class="flex-grow-1">
                        <h5 class="alert-heading fw-bold mb-2">Connection Error</h5>
                        <p class="mb-3">Unable to connect to the relay controller. Please check:</p>
                        <ul class="mb-3">
                            <li>Your internet connection</li>
                            <li>Node-RED server is running</li>
                            <li>ESP8266 is powered on and connected</li>
                            <li>MQTT broker is accessible</li>
                        </ul>
                        <button class="btn btn-danger" onclick="retryConnection()">
                            <i class="bi bi-arrow-repeat"></i> Retry Connection
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;
}

async function retryConnection() {
    const btn = event.target;
    btn.disabled = true;
    btn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Retrying...';
    
    state.retryCount = 0;
    
    try {
        await fetchRelayStatus();
        showToast('success', 'Connection restored!');
    } catch (error) {
        showToast('error', 'Connection failed. Please try again.');
    } finally {
        btn.disabled = false;
        btn.innerHTML = '<i class="bi bi-arrow-repeat"></i> Retry Connection';
    }
}

// ============================================
// NOTIFICATIONS
// ============================================
function showToast(type, message) {
    const toastContainer = document.getElementById('toastContainer');
    if (!toastContainer) return;
    
    const toastId = `toast-${Date.now()}`;
    const bgClass = type === 'success' ? 'bg-success' : 
                    type === 'error' ? 'bg-danger' : 
                    type === 'warning' ? 'bg-warning' : 'bg-info';
    
    const icon = type === 'success' ? 'check-circle-fill' : 
                 type === 'error' ? 'x-circle-fill' : 
                 type === 'warning' ? 'exclamation-triangle-fill' : 'info-circle-fill';
    
    const toastHTML = `
        <div id="${toastId}" class="toast align-items-center text-white ${bgClass} border-0" role="alert">
            <div class="d-flex">
                <div class="toast-body d-flex align-items-center gap-2">
                    <i class="bi bi-${icon} fs-5"></i>
                    <span>${message}</span>
                </div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        </div>
    `;
    
    toastContainer.insertAdjacentHTML('beforeend', toastHTML);
    
    const toastElement = document.getElementById(toastId);
    const toast = new bootstrap.Toast(toastElement, { autohide: true, delay: 4000 });
    toast.show();
    
    toastElement.addEventListener('hidden.bs.toast', () => {
        toastElement.remove();
    });
}

// ============================================
// AUTO UPDATE
// ============================================
function startAutoUpdate() {
    if (state.updateInterval) {
        clearInterval(state.updateInterval);
    }
    
    state.updateInterval = setInterval(() => {
        if (state.isConnected) {
            fetchRelayStatus().catch(console.error);
        }
    }, CONFIG.UPDATE_INTERVAL);
}

function stopAutoUpdate() {
    if (state.updateInterval) {
        clearInterval(state.updateInterval);
        state.updateInterval = null;
    }
}

// ============================================
// SYSTEM TIME
// ============================================
function updateSystemTime() {
    const now = new Date();
    const timeStr = now.toLocaleTimeString('en-US', { 
        hour12: false,
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit'
    });
    
    const timeElement = document.getElementById('currentTime');
    if (timeElement) {
        timeElement.textContent = timeStr;
    }
}

// ============================================
// EVENT LISTENERS
// ============================================
function setupEventListeners() {
    // Page visibility change
    document.addEventListener('visibilitychange', () => {
        if (document.hidden) {
            stopAutoUpdate();
        } else {
            startAutoUpdate();
            fetchRelayStatus().catch(console.error);
        }
    });
    
    // Window focus
    window.addEventListener('focus', () => {
        fetchRelayStatus().catch(console.error);
    });
    
    // Online/Offline events
    window.addEventListener('online', () => {
        showToast('success', 'Internet connection restored');
        fetchRelayStatus().catch(console.error);
    });
    
    window.addEventListener('offline', () => {
        showToast('warning', 'Internet connection lost');
        updateConnectionStatus(false);
    });
}

// ============================================
// UTILITY FUNCTIONS
// ============================================
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Add CSS for rotation animation
const style = document.createElement('style');
style.textContent = `
    .rotate {
        animation: rotate 1s linear infinite;
    }
    @keyframes rotate {
        from { transform: rotate(0deg); }
        to { transform: rotate(360deg); }
    }
`;
document.head.appendChild(style);

// ============================================
// ERROR HANDLING
// ============================================
window.addEventListener('error', (event) => {
    console.error('Global error:', event.error);
});

window.addEventListener('unhandledrejection', (event) => {
    console.error('Unhandled promise rejection:', event.reason);
});

// ============================================
// EXPORT FOR DEBUGGING (Development only)
// ============================================
if (typeof window !== 'undefined') {
    window.DEBUG = {
        state,
        fetchRelayStatus,
        updateDashboard,
        CONFIG
    };
}
