# ESP8266 Relay Timer Dashboard - Complete Setup Guide

## 📋 Table of Contents
1. [System Overview](#system-overview)
2. [Architecture](#architecture)
3. [Prerequisites](#prerequisites)
4. [Node-RED Setup](#node-red-setup)
5. [Web Dashboard Deployment](#web-dashboard-deployment)
6. [ESP8266 Configuration](#esp8266-configuration)
7. [API Documentation](#api-documentation)
8. [Troubleshooting](#troubleshooting)

---

## 🎯 System Overview

This is a complete IoT relay timer control system with:
- **ESP8266**: Controls 5 relays with scheduled routines and manual override
- **RTC DS3231**: Real-time clock for accurate timing
- **MQTT**: Communication protocol between ESP8266 and Node-RED
- **Node-RED**: Backend server providing REST API and MQTT handling
- **Web Dashboard**: Modern, responsive UI for monitoring and control

---

## 🏗️ Architecture

```
ESP8266 (Relays + RTC)
    ↕️ MQTT
Node-RED Server (MQTT Broker + REST API)
    ↕️ HTTP/REST
Web Dashboard (HTML/CSS/JS)
```

### Communication Flow:
1. **ESP8266 → Node-RED**: Publishes relay status to `relay/status/{0-4}` every 2 minutes
2. **Node-RED → ESP8266**: Publishes commands to `relay/set_mode` and `rtc/set`
3. **Web Dashboard → Node-RED**: REST API calls for data and control
4. **Node-RED → Web Dashboard**: JSON responses with relay states

---

## ✅ Prerequisites

### Hardware:
- ESP8266 (NodeMCU/Wemos D1 Mini)
- DS3231 RTC Module
- 5-Channel Relay Module
- Power Supply

### Software:
- Node.js (v14 or higher)
- Node-RED installed
- Web browser (Chrome, Firefox, Safari)
- MQTT Broker access (already configured: 49.207.44.7)

---

## 🚀 Node-RED Setup

### Step 1: Import the Flow

1. Open Node-RED editor: `https://sensorsnodered.kalpatech.co.in/`
2. Click menu (☰) → Import → Clipboard
3. Copy and paste the contents of `nodered-flow.json`
4. Click "Import"

### Step 2: Verify MQTT Broker Configuration

The flow includes a pre-configured MQTT broker node:
- **Host**: 49.207.44.7
- **Port**: 1883
- **Username**: sanjeevani
- **Password**: S@njeevani12345

To verify:
1. Double-click any MQTT node in the flow
2. Click the pencil icon next to "Relay MQTT Broker"
3. Verify the settings match above
4. Click "Update" then "Done"

### Step 3: Deploy the Flow

1. Click the red "Deploy" button in the top-right
2. Wait for "Successfully deployed" message
3. Check that all nodes show "connected" status

### Step 4: Verify Endpoints

The following endpoints will be available:

| Method | Endpoint | Description |
|--------|----------|-------------|
| GET | `/api/relays/status` | Get all relay statuses |
| POST | `/api/relays/mode` | Set relay mode (auto/manual) |
| POST | `/api/relays/routines` | Update relay routines |
| POST | `/api/rtc/sync` | Sync RTC with system time |

Test the status endpoint:
```bash
curl https://sensorsnodered.kalpatech.co.in/api/relays/status
```

---

## 🌐 Web Dashboard Deployment

### Option 1: Deploy to Web Server

1. Upload files to your web server:
   - `dashboard.html`
   - `styles.css`
   - `app.js`

2. Configure web server (Apache/Nginx) to serve these files

3. Ensure CORS is enabled if serving from different domain

### Option 2: Local Development

1. Create a new folder for the dashboard
2. Copy the three files: `dashboard.html`, `styles.css`, `app.js`
3. Open `dashboard.html` in a web browser
4. Or use a local server:
   ```bash
   # Using Python 3
   python -m http.server 8000
   
   # Using Node.js http-server
   npx http-server -p 8000
   ```
5. Access: `http://localhost:8000/dashboard.html`

### Configuration Check

In `app.js`, verify the API_BASE_URL is correct:
```javascript
const API_BASE_URL = 'https://sensorsnodered.kalpatech.co.in';
```

---

## 🔌 ESP8266 Configuration

### Hardware Connections

#### Relay Connections:
- Relay 1 → D0 (GPIO16)
- Relay 2 → D5 (GPIO14)
- Relay 3 → D6 (GPIO12)
- Relay 4 → D7 (GPIO13)
- Relay 5 → D4 (GPIO2)

#### RTC DS3231:
- SDA → D2 (GPIO4)
- SCL → D1 (GPIO5)
- VCC → 3.3V
- GND → GND

### Software Setup

1. **Install Arduino IDE** with ESP8266 board support
   - File → Preferences → Additional Board Manager URLs:
   - Add: `http://arduino.esp8266.com/stable/package_esp8266com_index.json`

2. **Install Required Libraries**:
   - ESP8266WiFi (included)
   - ESP8266WebServer (included)
   - DNSServer (included)
   - EEPROM (included)
   - Wire (included)
   - RTClib (by Adafruit)
   - PubSubClient (by Nick O'Leary)
   - ArduinoJson (by Benoit Blanchon)

3. **Upload Code**:
   - Open `Esp8266_Enhanced_FIXED.ino`
   - Select Board: "NodeMCU 1.0 (ESP-12E Module)"
   - Select Port: Your ESP8266 COM port
   - Click Upload

### Initial Configuration

1. **Connect to ESP8266 Hotspot**:
   - SSID: `RelayTimer-Setup`
   - Password: `12345678`

2. **Configure WiFi**:
   - Open browser → Automatically redirects to setup page
   - Or manually go to: `http://192.168.4.1`
   - Enter your WiFi credentials
   - Click "Save & Connect"

3. **Device will restart and connect to your WiFi**

4. **Find the IP address**:
   - Check your router's DHCP client list
   - Or use Serial Monitor (115200 baud)

---

## 📡 API Documentation

### 1. Get Relay Status

**Endpoint**: `GET /api/relays/status`

**Response**:
```json
{
  "relays": [
    {
      "relay": 0,
      "state": true,
      "mode": "Auto",
      "routines": [
        {
          "on": "06:00",
          "off": "18:00",
          "enabled": true
        }
      ]
    },
    // ... 4 more relays
  ],
  "rtc": {
    "date": "03/11/2025",
    "time": "14:30:45"
  }
}
```

### 2. Set Relay Mode

**Endpoint**: `POST /api/relays/mode`

**Request**:
```json
{
  "relay": 0,
  "mode": "auto" // or "manual_on" or "manual_off"
}
```

**Response**:
```json
{
  "success": true,
  "message": "Mode updated"
}
```

### 3. Update Relay Routines

**Endpoint**: `POST /api/relays/routines`

**Request**:
```json
{
  "relay": 0,
  "routines": [
    {
      "on": "06:00",
      "off": "08:00",
      "enabled": true
    },
    {
      "on": "18:00",
      "off": "22:00",
      "enabled": true
    }
  ]
}
```

**Response**:
```json
{
  "success": true,
  "message": "Routines updated"
}
```

### 4. Sync RTC

**Endpoint**: `POST /api/rtc/sync`

**Request**:
```json
{
  "date": "2025-11-03",
  "time": "14:30"
}
```

**Response**:
```json
{
  "success": true,
  "message": "RTC synced"
}
```

---

## 🔍 MQTT Topics

### Published by ESP8266:

#### Relay Status (Every 2 minutes per relay)
- **Topic**: `relay/status/0` to `relay/status/4`
- **Payload**:
```json
{
  "relay": 0,
  "state": true,
  "mode": "Auto",
  "routines": [
    {
      "on": "06:00",
      "off": "18:00",
      "enabled": true
    }
  ],
  "time": "14:30:45",
  "date": "03/11/2025"
}
```

### Subscribed by ESP8266:

#### Set Relay Mode
- **Topic**: `relay/set_mode`
- **Payload**:
```json
{
  "relay": 0,
  "mode": "auto"
}
```

#### Set RTC Time
- **Topic**: `rtc/set`
- **Payload**:
```json
{
  "date": "2025-11-03",
  "time": "14:30"
}
```

---

## 🐛 Troubleshooting

### ESP8266 Issues

**Problem**: ESP8266 not connecting to WiFi
- **Solution**: 
  - Verify WiFi credentials are correct
  - Check WiFi signal strength
  - Ensure 2.4GHz WiFi (ESP8266 doesn't support 5GHz)
  - Reset and reconfigure via hotspot

**Problem**: Relays not switching
- **Solution**:
  - Check physical connections
  - Verify relay module power supply
  - Test relay module independently
  - Check pin assignments in code

**Problem**: RTC not working
- **Solution**:
  - Verify I2C connections (SDA, SCL)
  - Check RTC battery
  - Test I2C scanner sketch
  - Ensure pull-up resistors on I2C lines

### Node-RED Issues

**Problem**: MQTT not connecting
- **Solution**:
  - Verify MQTT broker is running
  - Check credentials in MQTT broker node
  - Test with MQTT client tool (MQTT.fx, MQTT Explorer)
  - Check firewall settings

**Problem**: API endpoints not responding
- **Solution**:
  - Verify flow is deployed
  - Check Node-RED debug panel for errors
  - Test endpoints with curl or Postman
  - Verify HTTP In nodes are configured correctly

### Web Dashboard Issues

**Problem**: Dashboard shows "Disconnected"
- **Solution**:
  - Check API_BASE_URL in app.js
  - Verify Node-RED server is accessible
  - Check browser console for errors
  - Ensure CORS is properly configured

**Problem**: Unable to control relays
- **Solution**:
  - Check network connectivity
  - Verify ESP8266 is online
  - Check MQTT connection status
  - Review Node-RED debug logs

---

## 📊 Dashboard Features

### Real-time Monitoring
- Live relay states (ON/OFF)
- Active relay count
- Auto vs Manual mode distribution
- RTC date and time display
- Connection status indicator

### Relay Control
- **Auto Mode**: Follows scheduled routines
- **Manual ON**: Force relay on
- **Manual OFF**: Force relay off

### Routine Management
- Add/Edit/Delete time routines
- Up to 10 routines per relay
- Enable/Disable individual routines
- Visual time picker interface

### RTC Management
- View current RTC time
- Sync with system time button
- Automatic time updates

---

## 🔐 Security Considerations

1. **Change Default Credentials**:
   - Update MQTT username/password
   - Change ESP8266 hotspot password
   - Use strong WiFi credentials

2. **Network Security**:
   - Use HTTPS for web dashboard
   - Enable MQTT over TLS
   - Implement authentication for API endpoints
   - Use VPN for remote access

3. **Physical Security**:
   - Secure ESP8266 in enclosure
   - Protect relay module from moisture
   - Use proper electrical isolation

---

## 📈 Performance Tips

1. **Reduce MQTT Traffic**:
   - Adjust publishing interval (currently 2 minutes)
   - Publish only on state changes

2. **Optimize Dashboard**:
   - Increase refresh interval if needed
   - Minimize API calls
   - Use WebSocket for real-time updates (future enhancement)

3. **ESP8266 Stability**:
   - Use quality power supply
   - Add capacitors for stability
   - Enable watchdog timer

---

## 🔄 System Workflow

### Normal Operation:
1. ESP8266 reads RTC time every loop cycle
2. Checks if current time matches any routine
3. Updates relay states accordingly
4. Publishes status to MQTT every 2 minutes
5. Node-RED stores status and serves to dashboard
6. Dashboard polls Node-RED every 5 seconds

### User Control:
1. User clicks button on dashboard
2. Dashboard sends API request to Node-RED
3. Node-RED publishes MQTT command to ESP8266
4. ESP8266 receives command and updates relay
5. ESP8266 publishes new status
6. Dashboard updates to reflect change

---

## 📞 Support

For issues or questions:
1. Check troubleshooting section
2. Review Node-RED debug panel
3. Check ESP8266 serial monitor
4. Verify all connections and configurations

---

## 📝 Version Information

- **ESP8266 Firmware**: Enhanced FIXED version
- **Node-RED Flow**: v1.0
- **Web Dashboard**: v1.0
- **Last Updated**: November 2025

---

## 🎉 Quick Start Checklist

- [ ] Node-RED flow imported and deployed
- [ ] MQTT broker connection verified
- [ ] Web dashboard files uploaded
- [ ] ESP8266 code uploaded
- [ ] WiFi configured on ESP8266
- [ ] RTC time synced
- [ ] Test relay control from dashboard
- [ ] Verify routines are working
- [ ] Check auto-update is functioning

---

**Congratulations! Your Relay Timer Dashboard is now fully operational! 🚀**
