# Quick Reference Card - ESP8266 Relay Timer Dashboard

## 🔗 Quick Links

### Access Points
- **Node-RED Editor**: https://sensorsnodered.kalpatech.co.in/
- **Web Dashboard**: [Your deployment URL]/dashboard.html
- **ESP8266 Hotspot**: RelayTimer-Setup (Password: 12345678)

### MQTT Broker
```
Host: 49.207.44.7
Port: 1883
User: sanjeevani
Pass: S@njeevani12345
```

---

## 📡 API Endpoints (Quick Reference)

```bash
# Get all relay statuses
GET https://sensorsnodered.kalpatech.co.in/api/relays/status

# Set relay mode
POST https://sensorsnodered.kalpatech.co.in/api/relays/mode
{
  "relay": 0,
  "mode": "auto" // or "manual_on", "manual_off"
}

# Update routines
POST https://sensorsnodered.kalpatech.co.in/api/relays/routines
{
  "relay": 0,
  "routines": [
    { "on": "06:00", "off": "18:00", "enabled": true }
  ]
}

# Sync RTC
POST https://sensorsnodered.kalpatech.co.in/api/rtc/sync
{
  "date": "2025-11-03",
  "time": "14:30"
}
```

---

## 🔌 ESP8266 Pin Mapping

```
Relay 1 → D0 (GPIO16)
Relay 2 → D5 (GPIO14)
Relay 3 → D6 (GPIO12)
Relay 4 → D7 (GPIO13)
Relay 5 → D4 (GPIO2)

RTC SDA → D2 (GPIO4)
RTC SCL → D1 (GPIO5)
```

---

## 📊 MQTT Topics

### ESP8266 Publishes (Every 2 min):
```
relay/status/0
relay/status/1
relay/status/2
relay/status/3
relay/status/4
```

### ESP8266 Subscribes:
```
relay/set_mode
rtc/set
```

---

## 🧪 Testing Commands

### Test MQTT Connection
```bash
# Subscribe to all relay status
mosquitto_sub -h 49.207.44.7 -p 1883 -u sanjeevani -P "S@njeevani12345" -t "relay/status/#"

# Publish mode change
mosquitto_pub -h 49.207.44.7 -p 1883 -u sanjeevani -P "S@njeevani12345" \
  -t "relay/set_mode" -m '{"relay":0,"mode":"manual_on"}'
```

### Test API with cURL
```bash
# Get status
curl https://sensorsnodered.kalpatech.co.in/api/relays/status

# Set mode
curl -X POST https://sensorsnodered.kalpatech.co.in/api/relays/mode \
  -H "Content-Type: application/json" \
  -d '{"relay":0,"mode":"auto"}'
```

### Test with JavaScript (Browser Console)
```javascript
// Get status
fetch('https://sensorsnodered.kalpatech.co.in/api/relays/status')
  .then(r => r.json())
  .then(console.log);

// Set mode
fetch('https://sensorsnodered.kalpatech.co.in/api/relays/mode', {
  method: 'POST',
  headers: {'Content-Type': 'application/json'},
  body: JSON.stringify({relay: 0, mode: 'manual_on'})
})
  .then(r => r.json())
  .then(console.log);
```

---

## 🐛 Debug Checklist

### ESP8266 Not Connecting?
1. Check serial monitor (115200 baud)
2. Verify WiFi credentials
3. Ensure 2.4GHz network
4. Check power supply (5V, min 1A)

### MQTT Not Working?
1. Check Node-RED MQTT node status
2. Verify credentials
3. Test with MQTT client
4. Check firewall/port 1883

### API Not Responding?
1. Verify Node-RED is deployed
2. Check debug panel
3. Test with curl/Postman
4. Verify CORS headers

### Dashboard Not Updating?
1. Check browser console
2. Verify API_BASE_URL
3. Check network tab
4. Clear browser cache

---

## 📝 Common Code Snippets

### ESP8266: Manual Relay Control via Serial
```cpp
// In Arduino IDE Serial Monitor (115200 baud)
// Send: R0:1  (Turn ON Relay 0)
// Send: R0:0  (Turn OFF Relay 0)
```

### Node-RED: Custom Function Example
```javascript
// Get all relay states
let relayData = flow.get('relayData') || {};
msg.payload = relayData;
return msg;
```

### JavaScript: Format Time Routine
```javascript
function formatRoutine(routine) {
  return {
    onHour: parseInt(routine.on.split(':')[0]),
    onMinute: parseInt(routine.on.split(':')[1]),
    offHour: parseInt(routine.off.split(':')[0]),
    offMinute: parseInt(routine.off.split(':')[1]),
    enabled: routine.enabled
  };
}
```

---

## 🔄 Relay Mode Mapping

| Mode String | MQTT Value | Display | Behavior |
|-------------|-----------|---------|----------|
| Auto | `auto` | Auto | Follow routines |
| Manual On | `manual_on` | Manual ON | Always ON |
| Manual Off | `manual_off` | Manual OFF | Always OFF |

---

## ⏰ Time Format Reference

### Web Dashboard Format:
```
Time: "HH:MM" (24-hour, e.g., "14:30")
Date: "YYYY-MM-DD" (e.g., "2025-11-03")
```

### ESP8266 Format:
```
Time: "HH:MM:SS" (e.g., "14:30:45")
Date: "DD/MM/YYYY" (e.g., "03/11/2025")
```

### Routine Format:
```json
{
  "onHour": 6,
  "onMinute": 0,
  "offHour": 18,
  "offMinute": 0,
  "enabled": true
}
```

---

## 🎨 Dashboard Color Codes

```css
Primary: #667eea (Purple-Blue)
Success: #28a745 (Green)
Danger: #dc3545 (Red)
Warning: #ffc107 (Yellow)
Info: #17a2b8 (Cyan)
```

---

## 📦 Required Libraries

### ESP8266 (Arduino IDE):
- ESP8266WiFi
- ESP8266WebServer
- DNSServer
- EEPROM
- Wire
- RTClib (Adafruit)
- PubSubClient (Nick O'Leary)
- ArduinoJson (v6.x)

### Node-RED Palette:
- node-red-contrib-mqtt-broker (built-in)
- No additional palettes required!

---

## 🚀 One-Command Deploys

### Start Local Web Server (Python):
```bash
cd /path/to/dashboard
python -m http.server 8000
# Access: http://localhost:8000/dashboard.html
```

### Start Local Web Server (Node.js):
```bash
cd /path/to/dashboard
npx http-server -p 8000 -c-1
# Access: http://localhost:8000/dashboard.html
```

---

## 🔢 Default Values

```javascript
// Relay Names
const RELAY_NAMES = [
  'Relay 1', 'Relay 2', 'Relay 3', 'Relay 4', 'Relay 5'
];

// Default Routines (Auto mode)
Relay 0: 06:00-08:00, 18:00-20:00
Relay 1: 07:00-09:00, 19:00-21:00
Relay 2: 08:00-10:00, 20:00-22:00
Relay 3: 09:00-11:00, 21:00-23:00
Relay 4: 10:00-12:00, 22:00-00:00

// Update Intervals
Dashboard refresh: 5 seconds
MQTT publish: 2 minutes per relay
Auto-save: On state change
```

---

## 📱 Browser Compatibility

✅ Chrome 90+
✅ Firefox 88+
✅ Safari 14+
✅ Edge 90+
✅ Mobile browsers (iOS Safari, Chrome Mobile)

---

## 🎯 Performance Benchmarks

```
ESP8266:
- Boot time: ~5 seconds
- WiFi connect: ~3-15 seconds
- MQTT connect: ~2 seconds
- Relay response: <100ms

Node-RED:
- API response: <50ms
- MQTT processing: <10ms
- Flow execution: <5ms

Dashboard:
- Initial load: <1 second
- Status update: <200ms
- Mode change: <500ms
```

---

## 💡 Pro Tips

1. **Use Chrome DevTools** for debugging (F12)
2. **Monitor Serial Output** for ESP8266 diagnostics
3. **Check Node-RED Debug Panel** for flow issues
4. **Use MQTT Explorer** for live MQTT monitoring
5. **Test APIs with Postman** before frontend integration
6. **Enable Browser Cache** for faster dashboard loads
7. **Use WiFi Scanner** to find ESP8266 hotspot
8. **Bookmark Node-RED** for quick access
9. **Save Flow Backups** before making changes
10. **Document Custom Changes** for future reference

---

## 📞 Emergency Commands

### Reset ESP8266 to Factory:
```
1. Power off ESP8266
2. Hold FLASH button
3. Power on
4. Release after 5 seconds
5. Reconnect to hotspot
```

### Clear Node-RED Flow Storage:
```javascript
// In Node-RED function node
flow.set('relayData', null);
return msg;
```

### Force Dashboard Refresh:
```javascript
// In browser console
localStorage.clear();
location.reload(true);
```

---

**Keep this card handy for quick troubleshooting! 📋**
